/*---------------------------------------------------------------------------+
| Spy Window Procedure
|
| Change History:
| ---------------
| $C1=1.01,06/14/89,smd,increase HWND from 4 to 6 digits
| $C2=1.01,06/14/89,smd,draw message list with specified color
| $C3=1.02,06/17/89,jvk,added logging of messages to a file
| $C4=1.02,06/17/89,jvk,fixed text overlap into scrollbars on 8513 drivers
| $C5=1.03,06/19/89,smd,change in HELP call
| $C6=1.03,06/20/89,smd,add FILTER.GROUP, delete FILTER.MOUSE/Hit
| $C7=1.10,06/24/89,jvk,changed logging (more events logged, autostart
|                       after logfile selection
| $C8=1.10,06/24/89,jvk,added command line parameter support
| $C9=1.11,06/26/89,jvk,added profile support
|
+---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------+
| Includes                                                                   |
+---------------------------------------------------------------------------*/

#include "pmspy.h"                      /* Resource symbolic identifiers*/


#define PM_11   0x0A0A                 /* WinQueryVersion for 1.1 */
#define PM_12   0x140A                 /* WinQueryVersion for 1.2 */

/*---------------------------------------------------------------------------+
| Localized items
+---------------------------------------------------------------------------*/

typedef struct                                  /* @C2A - template for owner draw item */
{
  CHAR        logCLR;                           /* logical color index */
  CHAR        szMsg[80];

} HOOK_DATA, *PHOOKDATA;

typedef struct
{
  LONG   lStartDelta,
         lTotalLength;

} COLUMN_DATA;

typedef enum
{
  ColHwnd,
  ColMsg,
  ColMp1,
  ColMp2,

  ColTotal

} COLUMNS;

typedef COLUMN_DATA     DRAW_COLUMNS[ColTotal];

typedef DRAW_COLUMNS  *PDRAW_COLUMNS;

/*---------------------------------------------------------------------------
| Determine Size of RECTL needed to draw a string
+---------------------------------------------------------------------------*/
LONG DetermineDrawWidth(HWND hWnd, PSZ pString)
{
  HPS        hPS;                           /* Presentation space handle    */
  RECTL      Rectl;

  /* get a PS */
  hPS = WinGetPS( hWnd );
  /* fix passed RECTL so "too big" for string */
  Rectl.xLeft = Rectl.yBottom = 0;
  Rectl.xRight = WinQuerySysValue(HWND_DESKTOP, SV_CXFULLSCREEN);
  Rectl.yTop   = WinQuerySysValue(HWND_DESKTOP, SV_CYFULLSCREEN);
  /* see how big a RECTL is really needed to contain this string */
  WinDrawText(hPS,
              -1,                       /* its a NULL terminated string.... */
              pString,
              &Rectl,
              SYSCLR_WINDOWSTATICTEXT,  /* Foreground color */
              SYSCLR_BACKGROUND,        /* Background color */
              DT_LEFT | DT_BOTTOM | DT_QUERYEXTENT);
  /* Free the PS */
  WinReleasePS( hPS );
  return( Rectl.xRight );
}

/*---------------------------------------------------------------------------
| Toggle menu 'check' on exclusive items
+---------------------------------------------------------------------------*/
static VOID ToggleMenuChecking(register USHORT decheck, register USHORT check)
{
  WinSendMsg(hwndMenu, MM_SETITEMATTR,        /* de-check first */
             MPFROM2SHORT(decheck, TRUE),
             MPFROM2SHORT(MIA_CHECKED, 0) );
  WinSendMsg(hwndMenu, MM_SETITEMATTR,        /* then check */
             MPFROM2SHORT(check, TRUE),
             MPFROM2SHORT(MIA_CHECKED, MIA_CHECKED) );
}

/*---------------------------------------------------------------------------
| Enable/Disable menu pulldown item
+---------------------------------------------------------------------------*/
static VOID EnableMenuPulldown(register USHORT idItem,
                               register BOOL bEnable)
{
  WinSendMsg(hwndMenu,                                     /* Menu window */
             MM_SETITEMATTR,                               /* Action */
             MPFROM2SHORT(idItem, TRUE),                   /* Go look for it! */
             MPFROM2SHORT(MIA_DISABLED,
                          bEnable ? 0 : MIA_DISABLED) );   /* do right thing */
}

/*---------------------------------------------------------------------------
| Clear the Message List
+---------------------------------------------------------------------------*/
static VOID ClearMessageList( VOID )
{
  WinSendMsg(hwndMsgList, LM_DELETEALL, (MPARAM)NULL, (MPARAM)NULL);
  NbrOfMessages = 0;
}

/*---------------------------------------------------------------------------
| Draw Columnized Data
+---------------------------------------------------------------------------*/
VOID DrawColumnData(HPS           hPS,       /* PS to draw columns in */
                    PSZ           pText[],   /* array of ptrs to Text for columns */
                    COLUMN_DATA   ColData[], /* Column Data */
                    RECTL         recData,   /* RECTL to draw columns within */
                    LONG          uClrFill,  /* Rectangle Fill color */
                    LONG          uClrFG,    /* Column FG color */
                    LONG          uClrBG)    /* Column BG color */
{
  register INT         i;
           LONG        lxLeftBase,lxMaxRight;                     /* @C4C */

  WinFillRect(hPS,&recData,uClrFill);
  lxLeftBase   = recData.xLeft;
  lxMaxRight   = recData.xRight;                                  /* @C4A */
  for (i = 0; i < ColTotal; i++)   /* draw each column */ {
    recData.xLeft  = lxLeftBase    + ColData[i].lStartDelta;
    recData.xRight = recData.xLeft + ColData[i].lTotalLength;
    recData.xRight = min(recData.xRight, lxMaxRight);             /* @C4A */
    if (recData.xRight > recData.xLeft) {                         /* @C4A */
      WinDrawText(hPS,
                  -1, pText[i],
                  &recData,
                  uClrFG, uClrBG,
                  DT_LEFT | DT_VCENTER);
    } else {                                                      /* @C4A */
    } /* endif */                                                 /* @C4A */
  } /* endfor */
}

/*---------------------------------------------------------------------------+
| Spy Window Procedure Data                                                  |
+---------------------------------------------------------------------------*/
static COLUMN_DATA  ColPos[ColTotal] = {             /* Note: dynamically filled */
                                         { 0, 0 },
                                         { 0, 0 },
                                         { 0, 0 },
                                         { 0, 0 }
                                       };
static PSZ          ColHdr[ColTotal] = {             /* Note: dynamically filled */
                                         "hwnd",
                                         "msg",
                                         "mp1",
                                         "mp2",
                                       };
static CHAR          szFileName[FILEDLG_DRIVE_LENGTH + /* @C3 - dummy file var */
                                FILEDLG_PATH_LENGTH +  /* @C3 - dummy file var */
                                FILEDLG_FILE_LENGTH];  /* @C3 - dummy file var */

static HELP_DATA    helpActionBar = {                  /* @C5A - HELP dialog data */
                                      { NULL,
                                        IDT_HELP,
                                        IDT_AB_HELP,     /* HELP resource */
                                      },

                                      IDS_HELP_TITLE_MAIN
                                    };

static HELP_DATA    helpNews      = {                  /* @C5A - NEWS dialog data */
                                      { NULL,
                                        IDT_HELP,
                                        IDT_NEWS,        /* HELP resource */
                                      },

                                      IDS_HELP_TITLE_NEWS
                                    };

static HELP_DATA    helpDefaults  = {
                                      { NULL,
                                        IDT_HELP,
                                        IDT_DEFAULTS,    /* HELP resource */
                                      },

                                      IDS_HELP_TITLE_DEFAULTS
                                    };

static union { ULONG     u;
               QVERSDATA q;
             } versionPM = {NULL};

/*---------------------------------------------------------------------------
| Open/Append a Profile
+---------------------------------------------------------------------------*/

typedef enum {UseDefault, PromptForFile, UsePassedFile} PPF_MODE;

VOID ProcessProfileFile(HWND              hwndCur,
                        HWND              hwndFrame,
                        PSPY_DATA         pSpyData,
                        register BOOL     bOpen,  /* T=Open, F=Append */
                        register PPF_MODE mode,
                        PSZ               useName)

{
  register USHORT i;
           MSG    lowMsg, highMsg;

  /* Prompt user for PROFILE name IFF we need to.... */

  switch( mode )
  {
    case PromptForFile:
         if (GetFileSelection(hwndCur,
                              FILEDLG_OPEN,
                              szProPattern,szProDrive,szProPath,szProFile) < 0)
           return;

         strcpy(szFileName, szProDrive);

         if (szProPath[0] != '\0')
         {
           strcat(szFileName, "\\");
           strcat(szFileName, szProPath);
         }

         strcat(szFileName, "\\");
         strcat(szFileName, szProFile);
   break;

   default:    /* nothing to do.... */ ;
  }

  /*** reset SPY controls IFF open ***/

  if (bOpen)
    ResetSpyData(pSpyData);

  /*** Load the appropriate Profile ***/

  switch( mode )
  {
    case UsePassedFile:
         OpenProfile(hwndCur, hwndFrame, pSpyData, useName);
         break;

    case PromptForFile:
         OpenProfile(hwndCur, hwndFrame, pSpyData, szFileName);
         break;

    default:
         OpenProfile(hwndCur, hwndFrame, pSpyData, NULL);
         break;
  }

  /*** Pass new Message Range to DLL servant ***/

  if (pSpyData->TotalMsgs != 0)
  {
    /****************************************************************
    * Process all of the MSG slots
    ****************************************************************/
    for(/* Initialize */ i       = 0,         // @ first MSG_SLOT
                         lowMsg  = 0xFFFF,    // outrageous LOW
                         highMsg = 0;         // outrageous HIGH

        /* While      */ i < NO_MSG_SLOTS;    // while not all examined

        /* Iterate    */ i++)
    {
      /****************************************************************
      * Check for range change IFF message(s) in this slot
      ****************************************************************/
      if (pSpyData->Msgs[i].uNoMsgs != 0)
      {
        if (pSpyData->Msgs[i].MsgLow < lowMsg)     // new LOW range?
          lowMsg = pSpyData->Msgs[i].MsgLow;

        if (pSpyData->Msgs[i].MsgHigh > highMsg)   // new HIGH range?
          highMsg = pSpyData->Msgs[i].MsgHigh;
      }
    }

    /****************************************************************
    * Set new MSG range to optimize processing
    ****************************************************************/
    SpySetTargetMsgRange(SpyInstance, lowMsg, highMsg);
  }
}

/*---------------------------------------------------------------------------+
| Spy Window Procedure                                                       |
+---------------------------------------------------------------------------*/

MRESULT EXPENTRY SpyWindowProc(HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
{

  HPS                  hps;                    /* Presentation space handle    */
  RECTL                rectl,                  /* Window rectangle             */
                       rectl2;

  POWNERITEM           pOwn;

  LONG                 lTmpLen;

  register PMSG_ITEM   pMsg;

  HOOK_DATA            HookAddData, HookDrawData;     /* @C2A */

  INT                  rc;

  CHAR                 szText[256];

  HWND                 hwndTarget,
                       hwndP;

  HMQ                  hmqTarget;

  switch(msg)
  {
    /********************************************************************
    * Message from one of our "hook" procedures
    *
    * Mp1............PQMSG to data for "hooked" MSG
    * Mp2............NULL
    *
    * - decide if this instances should INCLUDE this MSG
    * - if so: a) format message
    *          b) add to list
    *          c) LOG if logging active
    *
    * Notes:
    * ------
    * - since this is the most time critical MSG, it should be left as
    *   the FIRST case of the switch statement!
    *
    ********************************************************************/

    case MSG_FROM_SPY_HOOK:
    {
      PQMSG pSpyMsg = (PQMSG) mp1;     // locate passed "hooked" data item

      /***************************************************************
      * Determine if MSG wanted
      *
      * NO if: 1) the message displaying is 'frozen' at the moment
      *        2) the message isn't found in the current profile
      ***************************************************************/

      if ( bSuspendMsgDisplay   ||                    // MSG display active?
           (WantMsg(pSpyData, pSpyMsg->msg) != Yes))  // INCLUDE this MSG?
        return( NULL );

      /***************************************************************
      * Include this "hooked" message
      *
      * - format MSG
      * - add to list
      ***************************************************************/

      pMsg = FormatMsg(pSpyData,
                       HookAddData.szMsg,
                       pSpyMsg->hwnd,
                       pSpyMsg->msg,
                       pSpyMsg->mp1,
                       pSpyMsg->mp2);

      HookAddData.logCLR = (CHAR)(pMsg->ClrFG + COLOR_BASE);              /* @C2A */

      /***************************************************************
      * is the SPY list now full?
      ***************************************************************/

      if (++NbrOfMessages == MaxNbrOfMessages) {
          NbrOfMessages--;                      /* Yes...delete "oldest" */
          WinSendMsg(hwndMsgList,
                     LM_DELETEITEM,
                     MPFROMSHORT(0),
                     NULL);
      } /* endif */

      /***************************************************************
      * insert the message at the end of the list (since time ordered)
      ***************************************************************/

      WinSendMsg(hwndMsgList,
                 LM_INSERTITEM,
                 MPFROMSHORT(LIT_END),
                 MPFROMP(&HookAddData));         /* @C2C */

      WinSendMsg(hwndMsgList,
                 LM_SETTOPINDEX,
                 MPFROMSHORT(NbrOfMessages),
                 NULL);

      /***************************************************************
      * are we logging to a file?
      ***************************************************************/
      if (bCurrentlyLogging)
        fprintf(pLogFile,
                "%8.8lX %-30.30s %8.8lX %8.8lX\n",
                pSpyMsg->hwnd,
                pMsg   ->pDesc,        // text version of binary MSG
                pSpyMsg->mp1,
                pSpyMsg->mp2);
    }
    break;

    /********************************************************************
    * Frame has been re-sized...adjust client area size to fit within it
    ********************************************************************/
    case WM_SIZE:
      WinQueryWindowRect(hwnd,&rectl);
      WinSetWindowPos(hwndMsgList,
                      HWND_TOP,
                      0,0,
                      (SHORT)rectl.xRight,
                      (SHORT)rectl.yTop - (SHORT)metrics.lMaxBaselineExt,
                      SWP_SIZE | SWP_MOVE);
      return(NULL);
      break;

    /********************************************************************
    * the user is trying to CLOSE us...prompt if actively SPYing
    ********************************************************************/
    case WM_CLOSE:

      if ( (SpyQueryTargetWindow(SpyInstance) != SPY_BAD_HWND ) &&
           ( MsgDisplay( hwnd,
                         swcntrl.szSwtitle,
                         Strings[IDS_FMT_OK_TO_EXIT],
                         0,
                         MB_CUAWARNING | MB_OKCANCEL
                       ) != MBID_OK) )
        return(NULL); /* resume application */

      SpyUnSetTarget(SpyInstance);

      /* close logfile if still open */                         /* @C3A */

      if (pLogFile) {                                           /* @C3A */
        if (bCurrentlyLogging)                                  /* @C7A */
          fputs("---- exit PMSPY ---- \n", pLogFile);           /* @C7A */
        fclose(pLogFile);                                       /* @C3A */
      }

      /* tell main window to terminate */

      WinPostMsg(hwndFrame, PMSPY_QUIT_NOTICE, (ULONG) NULL, (ULONG) NULL);

      return(NULL);
      break;

    /********************************************************************
    * This message is sent each time a PULLDOWN menu is to be used
    *
    * - we dynamically enable/disable various items based on our 'state'
    ********************************************************************/

    case WM_INITMENU:
      switch (LOUSHORT(mp1))
      {
        case ID_AB_SPYON:
          if (SpyQueryTargetWindow(SpyInstance) == SPY_BAD_HWND )
          {
            ToggleMenuChecking(ID_SELECT,   ID_DESELECT);  /* nothing SPYed */
            EnableMenuPulldown(ID_SELECT,   TRUE);
            EnableMenuPulldown(ID_DESELECT, FALSE);
          }
          else
          {
            ToggleMenuChecking(ID_DESELECT, ID_SELECT);
            EnableMenuPulldown(ID_SELECT,   TRUE);
            EnableMenuPulldown(ID_DESELECT, TRUE);
          }
          break;

        case ID_AB_SPYWHAT:
          if ( SpyQueryTargetIsWindow(SpyInstance)  )
            ToggleMenuChecking(IDD_QUE, IDD_WND);
          else
            ToggleMenuChecking(IDD_WND, IDD_QUE);
          break;

        case ID_AB_FILE:

          ToggleMenuChecking(pLogFile == NULL ? ID_F_OPENLOG   : ID_F_CLOSELOG,
                             pLogFile == NULL ? ID_F_CLOSELOG  : ID_F_OPENLOG);

          EnableMenuPulldown(ID_F_OPENLOG,  pLogFile == NULL);
          EnableMenuPulldown(ID_F_CLOSELOG, pLogFile != NULL);

          break;

        case ID_AB_OPTIONS:
          ToggleMenuChecking(bSuspendMsgDisplay ? ID_O_THAW    : ID_O_FREEZE,
                             bSuspendMsgDisplay ? ID_O_FREEZE  : ID_O_THAW  );
          EnableMenuPulldown(ID_O_THAW,    bSuspendMsgDisplay);
          EnableMenuPulldown(ID_O_FREEZE, !bSuspendMsgDisplay);

          if (pLogFile != NULL) /* LOGing active? */
          {
            ToggleMenuChecking(bCurrentlyLogging ? ID_O_STOPLOG  : ID_O_STARTLOG,
                               bCurrentlyLogging ? ID_O_STARTLOG : ID_O_STOPLOG);
            EnableMenuPulldown(ID_O_STARTLOG, !bCurrentlyLogging);
            EnableMenuPulldown(ID_O_STOPLOG,   bCurrentlyLogging);
          }
          else
          {
            EnableMenuPulldown(ID_O_STARTLOG, FALSE);
            EnableMenuPulldown(ID_O_STOPLOG,  FALSE);
          }
         break;

        case ID_AB_FILTER:
          break;
      } /* endswitch */
      break;

    /********************************************************************
    * This message is sent each time a PULLDOWN menu is selected
    ********************************************************************/
    case WM_COMMAND:
      switch (SHORT1FROMMP(mp1)) {
        case ID_F_EXIT:
          WinPostMsg(hwnd, WM_CLOSE, 0L, 0L);
          break;

        case ID_H_ABOUT:
          WinDlgBox(HWND_DESKTOP, hwnd, (PFNWP)SpyLogoDlgProc,
                    NULL, IDD_ABOUT, NULL);

          break;

        case ID_H_NEWS:
          WinDlgBox(HWND_DESKTOP, hwnd, (PFNWP)HelpWindowProc,      /* @C5C */
                    NULL, IDD_HELP, (PVOID)&helpNews);
          break;

        case ID_H_BASICS:
          WinDlgBox(HWND_DESKTOP, hwnd, (PFNWP)HelpWindowProc,      /* @C5C */
                    NULL, IDD_HELP, (PVOID)&helpActionBar);
          break;

        case ID_SELECT:
          /* capture the mouse */
          bSelecting = TRUE;
          SpyUnSetTarget(SpyInstance);
          ClearMessageList();

          WinSetCapture(HWND_DESKTOP, hwnd);
          hOld = WinQueryPointer(HWND_DESKTOP);
          WinSetPointer(HWND_DESKTOP, hSpy);

          /* force the window in the background */
          WinSetWindowPos(hwndFrame, HWND_BOTTOM, 0, 0, 0, 0, SWP_ZORDER);
          break;

       case ID_SELECT_OBJECT:
          WinDlgBox( HWND_DESKTOP,
                     hwnd,
                     ObjectDlgProc,
                     NULL,
                     ID_OW_DLG,
                     NULL);
          break;

        case ID_DESELECT:
          SpyUnSetTarget(SpyInstance);
          bWindowIsObject = FALSE;
          UpdateSpyTitle(hwndFrame, tStandard);
          break;

        case ID_O_CLEAR:
          ClearMessageList();
          break;

        case ID_O_THAW:
          bSuspendMsgDisplay = FALSE;
          if (bCurrentlyLogging) {                                  /* @C3A */
            fputs("---- thaw list ---- \n", pLogFile);              /* @C3A */
          } else {                                                  /* @C3A */
          } /* endif */                                             /* @C3A */
          break;
        case ID_O_FREEZE:
          bSuspendMsgDisplay = TRUE;
          if (bCurrentlyLogging) {                                  /* @C3A */
            fputs("---- freeze list ---- \n", pLogFile);            /* @C3A */
          } else {                                                  /* @C3A */
          } /* endif */                                             /* @C3A */
          break;

       case ID_F_CLOSELOG:
          WinSendMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_O_STOPLOG), NULL);
          fclose(pLogFile);                                       /* @C3A */
          pLogFile = (FILE *)NULL;                                /* @C3A */
          break;

        case ID_F_OPENLOG:                                          /* @C3A */

          if( (rc = GetFileSelection(hwnd,
                                     FILEDLG_SAVE,
                                     szLogPattern,
                                     szLogDrive, szLogPath, szLogFile)) < 0)           /* use cancelled... */
            break;               /* User cancelled operation... */

          strcpy(szFileName, szLogDrive);                         /* @C3A */
          if (szLogPath[0] != '\0') {                             /* @C3A */
            strcat(szFileName, "\\");                             /* @C3A */
            strcat(szFileName, szLogPath);                        /* @C3A */
          }                                                       /* @C3A */
          strcat(szFileName, "\\");                               /* @C3A */
          strcat(szFileName, szLogFile);                          /* @C3A */

          /* if file exists but they don't want to replace, quit */
          if ( (rc == 0) &&                                       /* @C3A */
               (MsgDisplay(hwnd,
                           swcntrl.szSwtitle,
                           Strings[IDS_MSG_LOG_EXISTS],
                           0,
                           MB_ICONQUESTION | MB_YESNO | MB_MOVEABLE,
                           szFileName ) != MBID_YES) )                                 /* @C3A */
            break;

          pLogFile = fopen(szFileName, "w");                        /* @C3A */

          /**** N O T E:  falling through......                     // @C7D */

        case ID_O_STARTLOG:                                         /* @C3A */
          bCurrentlyLogging = TRUE;                                 /* @C3A */
          if (SpyQueryTargetWindow(SpyInstance) != SPY_BAD_HWND )
          {
            fputs("---- Start Spy Log ---- ", pLogFile);            /* @C3A */
            if (SpyQueryTargetIsWindow(SpyInstance)) {              /* @C3A */
              fprintf(pLogFile, Strings[IDS_FMT_SPY_WINDOW],        /* @C3A */
                      SpyQueryTargetWindow(SpyInstance));
            } else {                                                /* @C3A */
              fprintf(pLogFile, Strings[IDS_FMT_SPY_QUEUE],         /* @C3A */
                      SpyQueryTargetQueue(SpyInstance));
            } /* endif */                                           /* @C3A */
            fputc('\n', pLogFile);                                  /* @C3A */
          } else {                                                  /* @C3A */
          } /* endif */                                             /* @C3A */
          break;                                                    /* @C3A */
        case ID_O_STOPLOG:                                          /* @C3A */
          bCurrentlyLogging = FALSE;                                /* @C3A */
          fputs("---- Stop Spy Log ---- \n", pLogFile);             /* @C3A */
          break;                                                    /* @C3A */

        case ID_F_APPENDPROFILE:
          ProcessProfileFile(hwnd, hwndFrame, pSpyData, FALSE, PromptForFile, NULL);
          break;

        case ID_F_OPENPROFILE:                                      /* @C9A */
          ProcessProfileFile(hwnd, hwndFrame, pSpyData, TRUE,  PromptForFile, NULL);
          break;                                                    /* @C9A */

        case ID_F_SAVEPROFILE:                                      /* @C9A */
          if ((rc = GetFileSelection(hwnd,
                                     FILEDLG_SAVE,
                                     szProPattern,
                                     szProDrive, szProPath, szProFile)) < 0)
            break;

          strcpy(szFileName, szProDrive);                         /* @C9A */

          if (szProPath[0] != '\0') {                             /* @C9A */
            strcat(szFileName, "\\");                             /* @C9A */
            strcat(szFileName, szProPath);                        /* @C9A */
          }

          strcat(szFileName, "\\");                               /* @C9A */
          strcat(szFileName, szProFile);                          /* @C9A */

          if ( (rc == 0) &&                                       /* @C3A */
               (MsgDisplay(hwnd,
                           swcntrl.szSwtitle,
                           Strings[IDS_MSG_PROFILE_EXISTS],
                           0,
                           MB_ICONQUESTION | MB_YESNO | MB_MOVEABLE,
                           szFileName ) != MBID_YES) )                                 /* @C3A */
            break;

          SaveProfile(hwnd, pSpyData, szFileName);                  /* @C9A */
          break;                                                    /* @C9A */

        case ID_F_OPENDEFAULT:
            ProcessProfileFile(hwnd, hwndFrame, pSpyData, TRUE,  UseDefault, NULL);
          break;

        case ID_F_APPENDDEFAULT:
            ProcessProfileFile(hwnd, hwndFrame, pSpyData, FALSE, UseDefault, NULL);
          break;

        case ID_O_BROWSEDEFAULT:
          WinDlgBox(HWND_DESKTOP, hwnd, (PFNWP)HelpWindowProc,
                    NULL, IDD_HELP, (PVOID)&helpDefaults);

          break;

        case ID_F_INCLUDE:
          IncludeAllMsgs( pSpyData, TRUE,  Color_Asis );
          break;

        case ID_F_EXCLUDE:
          IncludeAllMsgs( pSpyData, FALSE, Color_Asis );
          break;

        case ID_F_SELECTIVE:
           WinDlgBox( HWND_DESKTOP,
                      hwnd,
                      (PFNWP)FilterDlgProc,
                      (HMODULE)NULL,
                      IDD_FILTER,
                      NULL);
          break;

        case ID_F_GROUP:                                            /* @C5A */
           WinDlgBox( HWND_DESKTOP,                                 /* @C5A */
                      hwnd,
                      (PFNWP)GroupDlgProc,
                      (HMODULE)NULL,
                      ID_GRP_DLG,
                      NULL);
          break; /* @C5A */

        case IDD_WND:
          SpySetTargetIsWindow(SpyInstance, TRUE);

          if ( SpyQueryTargetWindow(SpyInstance) != SPY_BAD_HWND)
            UpdateSpyTitle(hwndFrame,
                           bWindowIsObject ? tObject : tWindow,
                           SpyQueryTargetWindow(SpyInstance));
        break;

        case IDD_QUE:
          SpySetTargetIsWindow(SpyInstance, FALSE);

          if ( SpyQueryTargetQueue(SpyInstance) != SPY_BAD_HMQ)
            UpdateSpyTitle(hwndFrame,
                           tQueue,
                           SpyQueryTargetQueue(SpyInstance));

        break;

        default:
          return WinDefWindowProc( hwnd, msg, mp1, mp2 );
      } /* endswitch */
      break;

    /********************************************************************
    * This message is sent each time a PULLDOWN menu is selected
    ********************************************************************/
    case WM_MOUSEMOVE:
      break;

    /********************************************************************
    * User pressed mouse button #1..
    *
    * - if they are 'selecting' a window, determine which window is now
    *   under the POINTER
    ********************************************************************/
    case WM_BUTTON1DOWN: /* select for spying */

      if (bSelecting) {
        bSelecting = FALSE;        WinSetPointer(HWND_DESKTOP, hOld);
        WinSetCapture(HWND_DESKTOP, NULL);
        ptrPos.x = (LONG)SHORT1FROMMP(mp1);
        ptrPos.y = (LONG)SHORT2FROMMP(mp1);
        WinMapWindowPoints(hwnd, HWND_DESKTOP, (PPOINTL)&ptrPos, 1);
        hwndTarget = WinWindowFromPoint(HWND_DESKTOP, (PPOINTL)&ptrPos,
                                      TRUE, FALSE);
        hmqTarget  = (HMQ)WinQueryWindowULong(hwndTarget, QWL_HMQ);

        // trying to "spy" on the SPY?

        if (hmqTarget == hmq) {
          MsgDisplay( hwnd,
                      swcntrl.szSwtitle,
                      Strings[IDS_FMT_SPY_THE_SPY],
                      0,
                      MB_CUAWARNING | MB_CANCEL );
          UpdateSpyTitle(hwndFrame, tStandard); /* reset Window Title */
        }
        else {
          SpySetTarget(SpyInstance, hwndTarget, hmqTarget);
          bWindowIsObject = FALSE;

          UpdateSpyTitle(hwndFrame,
                         SpyQueryTargetIsWindow(SpyInstance)
                         ? tWindow
                         : tQueue,
                         SpyQueryTargetIsWindow(SpyInstance)
                         ? hwndTarget
                         :  hmqTarget);
        } /* endif */
      }
      else {
        return(WinDefWindowProc(hwnd, msg, mp1, mp2));
      } /* endif */
      break;

    /********************************************************************
    * User pressed mouse button #2..
    *
    * - if they are 'selecting' a window, display the window class, handle,
    *   and queue that is under the POINTER
    ********************************************************************/

    case WM_BUTTON2DOWN:  /* display hwnd and hmq */

      if (bSelecting)
      {
        ptrPos.x = (LONG)SHORT1FROMMP(mp1);
        ptrPos.y = (LONG)SHORT2FROMMP(mp1);

        WinMapWindowPoints(hwnd, HWND_DESKTOP, (PPOINTL)&ptrPos, 1);

        hwndTarget = WinWindowFromPoint(HWND_DESKTOP,
                                        (PPOINTL)&ptrPos, TRUE, FALSE);
         hmqTarget = (HMQ)WinQueryWindowULong(hwndTarget, QWL_HMQ);

        WinQueryClassName(hwndTarget, sizeof(szText), szText);

        UpdateSpyTitle(hwndFrame,
                       tSpyWho,
                       TranslateClassName(szText), hwndTarget, hmqTarget);
      }
      else
        return(WinDefWindowProc(hwnd, msg, mp1, mp2));
    break;

    /********************************************************************
    * Re-Paint the client area
    ********************************************************************/

    case WM_PAINT:
      hps = WinBeginPaint( hwnd, NULL, &rectl );
      WinQueryWindowRect( hwnd,        &rectl );    /* size of client area */
      WinQueryWindowRect( hwndMsgList, &rectl2 );   /* size of Msg List    */
      WinSetRect( hab,
                  &rectl,
                  0,
                  (SHORT)rectl2.yTop - 1,    /* overlap top of list a bit... */
                  (SHORT)rectl.xRight,
                  (SHORT)rectl.yTop);
      DrawColumnData(hps,
                     &ColHdr[0],
                     &ColPos[0],
                     rectl,
                     SYSCLR_WINDOW,
                     SYSCLR_WINDOWSTATICTEXT,
                     SYSCLR_WINDOW);
      WinEndPaint( hps );
      return(NULL);
      break;

    /********************************************************************
    * Provide size of item to be drawn in the SPY message list
    *
    * - this is part of the responsibility of an OWNERDRAW listbox
    ********************************************************************/

    case WM_MEASUREITEM:
      hps = WinGetPS(hwnd);
      GpiQueryFontMetrics(hps, (LONG)sizeof(FONTMETRICS), &metrics );
      WinReleasePS(hps);
      return(MPFROMLONG(metrics.lMaxBaselineExt));
      break;

    /********************************************************************
    * Draw the specified item in the SPY message list
    *
    * - this is part of the responsibility of an OWNERDRAW listbox
    ********************************************************************/

    case WM_DRAWITEM:  /* draw a listbox item */
      pOwn = (POWNERITEM)mp2;
      if ( (pOwn->fsState == pOwn->fsStateOld) ||  /* not a selection change */
           (pOwn->fsStateOld && pOwn->fsState) ) {
     /* @C2D   CHAR  szLBText[128]; */
        PSZ   msgText[ColTotal];
        WinSendMsg(pOwn->hwnd,
                   LM_QUERYITEMTEXT,
                   MPFROM2SHORT(pOwn->idItem, sizeof(HookDrawData)),
                   (MPARAM)&HookDrawData);                              /* @C2C */
        msgText[ColHwnd] = strtok(HookDrawData.szMsg, "\15");           /* @C2C */
        msgText[ColMsg ] = strtok(NULL,               "\15");
        msgText[ColMp1 ] = strtok(NULL,               "\15");
        msgText[ColMp2 ] = strtok(NULL,               "\15");
        DrawColumnData(pOwn->hps,
                       &msgText[0],
                       &ColPos[0],
                       pOwn->rclItem,
                       SYSCLR_WINDOW,
ExternalColorTranslation[pSpyData->LogicalColorTranslation[HookDrawData.logCLR-COLOR_BASE].iExtColor].lClrValue, /* @C2C */
                       SYSCLR_WINDOW);
      } /* endif */
      return((MRESULT)TRUE);  /* Keep PM from drawing it too */
      break;

    /********************************************************************
    * Special one-time initialization processing
    ********************************************************************/

    case WM_CREATE:
      /* Get information about which version of PM we're running on */
      versionPM.u = WinQueryVersion(hab);

      /* Get information about FONT being used */
      hps = WinGetPS(hwnd);
      GpiQueryFontMetrics(hps, (LONG)sizeof(metrics), &metrics);
      WinReleasePS(hps);

      /* Load the 'default' PROFILE */

      ProcessProfileFile(hwnd, hwndFrame, pSpyData, TRUE, UseDefault, NULL);

      /* Calculate column sizing data */

      ColHdr[ColHwnd] = Strings[IDS_HDR_HWND];
      ColHdr[ColMsg ] = Strings[IDS_HDR_MSG];
      ColHdr[ColMp1 ] = Strings[IDS_HDR_MP1];
      ColHdr[ColMp2 ] = Strings[IDS_HDR_MP2];

      /* determine length of various items we'll need to draw later */

      if ( versionPM.u < PM_12 )   /* Handle size is based on PM version! */
      {
        ColPos[ColHwnd].lTotalLength = DetermineDrawWidth(hwnd, "DDDD");
        pszFmtStd = Strings[IDS_FMT_STANDARD];
      }
      else
      {
        ColPos[ColHwnd].lTotalLength = DetermineDrawWidth(hwnd, "DDDDDDDD");
        pszFmtStd = Strings[IDS_FMT_STANDARD_12];
      }

      ColPos[ColMp1 ].lTotalLength =
      ColPos[ColMp2 ].lTotalLength = DetermineDrawWidth(hwnd, "DDDDDDDD");

      for( /* Initialize */  pMsg = ProcessFirstMsg(pSpyData);  /* start @ first MSG */
           /* Terminate  */  pMsg != NULL;      /* stop at End-Of-Table */
           /* Iterate    */  pMsg = ProcessNextMsg(pSpyData)  /* try the next MSG */
         ) {
         if ( (lTmpLen = DetermineDrawWidth(hwnd, pMsg->pDesc)) >
               ColPos[ColMsg].lTotalLength)
           ColPos[ColMsg].lTotalLength = lTmpLen;

         xMaxMsgLength = max(xMaxMsgLength, strlen(pMsg->pDesc));  /* @C3A */
      } /* endfor */

      /* use some % of widest WM_xxx message (from .DLL) */

      ColPos[ColMsg].lTotalLength = (ColPos[ColMsg].lTotalLength *
                                     atoi(Strings[IDS_MAX_MSG_PERCENT])) / 100;
      ColPos[ColMsg].lStartDelta = ColPos[ColHwnd].lStartDelta  +
                                   ColPos[ColHwnd].lTotalLength;
      ColPos[ColMp1].lStartDelta = ColPos[ColMsg ].lStartDelta  +
                                   ColPos[ColMsg ].lTotalLength;
      ColPos[ColMp2].lStartDelta = ColPos[ColMp1 ].lStartDelta  +
                                   ColPos[ColMp1 ].lTotalLength;

      hwndP = WinQueryWindow(hwnd, QW_PARENT, FALSE);
      hwndMsgList=WinCreateWindow( hwnd,
                                   WC_LISTBOX,
                                   "PM_Spy_List",
                                   WS_VISIBLE   |
                                   LS_OWNERDRAW | WS_TABSTOP | LS_NOADJUSTPOS,
                                   0, 0, 0, 0,
                                   hwnd,
                                   HWND_TOP,
                                   2,
                                   NULL,
                                   NULL);
      hwndMenu = WinWindowFromID(hwndP, FID_MENU);                     /* @C9M */
      MaxNbrOfMessages = atoi(Strings[IDS_MAX_MESSAGES]);              /* @C9M */
      bSelecting = FALSE;                                              /* @C9M */

      /* default to "Thawed" list */                                   /* @C9M */
      bSuspendMsgDisplay = FALSE;                                      /* @C9M */

      /* Size Frame Window so data just fits within scroll bar area */

      { INT  x,y,cx,cy,j,bMin;                                         /* @C8A */

        bMin = FALSE;                                                  /* @C8A */
        x  = 0;                                                        /* @C8A */
        y  = (SHORT)WinQuerySysValue(HWND_DESKTOP, SV_CYICON) * 2;     /* @C8A */
        /* size so columns fit and half height of screen */            /* @C8A */
        cx = (SHORT)(ColPos[ColMp2].lStartDelta  +                     /* @C8A */
                     ColPos[ColMp2].lTotalLength +                     /* @C8A */
                     WinQuerySysValue(HWND_DESKTOP, SV_CXVSCROLL));    /* @C8A */
        cy = (SHORT)WinQuerySysValue(HWND_DESKTOP, SV_CYFULLSCREEN)/2; /* @C8A */

        /* parse command line */                                       /* @C8A */
        for (j=1; j<ArgC; j++) {                                       /* @C8A */
          strupr((*ArgV)[j]);                                          /* @C8A */
          if (strncmp((*ArgV)[j], "/X=", 3) == 0) {                    /* @C8A */
            x  = atoi((*ArgV)[j]+3);                                   /* @C8A */
            x  = min(x,(SHORT)WinQuerySysValue(HWND_DESKTOP, SV_CXFULLSCREEN));
            x  = max(x,0);                                             /* @C8A */
          } else if (strncmp((*ArgV)[j], "/Y=", 3) == 0) {             /* @C8A */
            y  = atoi((*ArgV)[j]+3);                                   /* @C8A */
            y  = min(y,(SHORT)WinQuerySysValue(HWND_DESKTOP, SV_CYFULLSCREEN));
            y  = max(y,0);                                             /* @C8A */
          } else if (strncmp((*ArgV)[j], "/CX=", 4) == 0) {            /* @C8A */
            cx = atoi((*ArgV)[j]+4);                                   /* @C8A */
            cx = min(cx,(SHORT)WinQuerySysValue(HWND_DESKTOP, SV_CXFULLSCREEN)-x);
            cx = max(cx,0);                                            /* @C8A */
          } else if (strncmp((*ArgV)[j], "/CY=", 4) == 0) {            /* @C8A */
            cy = atoi((*ArgV)[j]+4);                                   /* @C8A */
            cy = min(cy,(SHORT)WinQuerySysValue(HWND_DESKTOP, SV_CYFULLSCREEN)-y);
            cy = max(cy,0);                                            /* @C8A */
          } else if (strncmp((*ArgV)[j], "/MIN", 4) == 0) {            /* @C8A */
            bMin = TRUE;                                               /* @C8A */
          } else if (strncmp((*ArgV)[j], "/LOG=", 5) == 0) {           /* @C8A */
            _splitpath((*ArgV)[j]+5, szLogDrive, szLogPath,            /* @C8A */
                       szLogFile, szLogPattern);                       /* @C8A */
            strcpy(szFileName, szLogDrive);                            /* @C8A */
            if (szLogPath[0] != '\0' && szLogPath[1] != '\0') {        /* @C8A */
              strcat(szFileName, "\\");                                /* @C8A */
              strcat(szFileName, szLogPath);                           /* @C8A */
            }                                                          /* @C8A */
            strcat(szLogFile, szLogPattern);                           /* @C9A */
            strcat(szFileName, szLogFile);                             /* @C8A */
            strcpy(szLogPattern, "*.LOG");                             /* @C8A */
            pLogFile = fopen(szFileName, "w");                         /* @C8A */
            if (pLogFile) {                                            /* @C8A */
              bCurrentlyLogging = TRUE;                                /* @C8A */
            }                                                          /* @C8A */
          } else if ( (strncmp((*ArgV)[j], "/PRO=", 5) == 0) ||        /* @C9A */
                      (strncmp((*ArgV)[j], "/APP=", 5) == 0) ) {
            _splitpath((*ArgV)[j]+5, szProDrive, szProPath,            /* @C9A */
                       szProFile, szProPattern);                       /* @C9A */
            strcpy(szFileName, szProDrive);                            /* @C9A */
            if (szProPath[0] != '\0' && szProPath[1] != '\0') {        /* @C9A */
              strcat(szFileName, "\\");                                /* @C9A */
              strcat(szFileName, szProPath);                           /* @C9A */
            }                                                          /* @C9A */
            strcat(szProFile, szProPattern);                           /* @C9A */
            strcat(szFileName, szProFile);                             /* @C9A */
            strcpy(szProPattern, "*.PRO");                             /* @C9A */

            ProcessProfileFile(hwnd,
                               hwndFrame,
                               pSpyData,
                               strncmp((*ArgV)[j], "/PRO=", 5) == 0,
                               UsePassedFile,szFileName);

          } else {                                                     /* @C8A */
            /* unknown command line parameter will be ignored */       /* @C8A */
          } /* endif */                                                /* @C8A */
        } /* endfor */                                                 /* @C8A */
        if (bMin) {                                                    /* @C8A */
          WinSetWindowPos(hwndP, HWND_TOP, x, y, cx, cy,               /* @C8A */
                          SWP_MOVE   | SWP_SIZE     | SWP_ZORDER |     /* @C8A */
                          SWP_MINIMIZE | SWP_SHOW);                    /* @C8A */
        } else {                                                       /* @C8A */
          WinSetWindowPos(hwndP, HWND_TOP, x, y, cx, cy,               /* @C8A */
                          SWP_MOVE   | SWP_SIZE     |                  /* @C8A */
                          SWP_ZORDER | SWP_ACTIVATE | SWP_SHOW);       /* @C8A */
        } /* endif */                                                  /* @C8A */
      }                                                                /* @C8A */

      WinSetFocus(HWND_DESKTOP, hwndMsgList);                          /* @C9M */
      return((MPARAM)NULL);
      break;

    /********************************************************************
    * We don't need to handle anyother messages, so let PM do it defaultly
    ********************************************************************/
    default:
      return(WinDefWindowProc(hwnd, msg, mp1, mp2));
  } /* endswitch */

  return((MPARAM)NULL);
}
